/*
   This program creates three child processes and then builds
   a three stage pipeline out of those three processes. This
   parent process is not part of the pipeline. It does not
   share in any of the work done in the pipeline.

   To build the pipeline, the parent starts each of the child
   processes, then it makes the first child inherit the parent's
   stdin, then it makes the third child inherit the parent's
   stdout, then it redirects the first child's stdout to the
   second child's stdin and it redirects the second child's
   stdout to the third child's stdin. The parent should end up
   sharing stdin with the first child and stdout with the third
   child, but the parent should make no use of stdin and stdout
   while its children are running.


   Important: Notice that this program is logically
   the same as the following command line pipe.

   C:\> java RemoveVowels | java ToUpperCase | java Reverse

   In the case of the above command line, the shell
   program (cmd.exe) sets up the pipeline before running
   the three programs. In the case of this program, the
   program itself creates the pipeline between the three
   child processes. So this program is acting like a very
   simple shell program.
*/
import java.io.*;

public class Java7_MultiPipeline
{
   public static void main(String[] args) throws IOException, InterruptedException
   {
      // Create a command line for running the stage3 child.
      ProcessBuilder pb3 = new ProcessBuilder("java", "Reverse");
      // Have the stage3 child inherit the parent's standard output stream.
      pb3.redirectOutput( ProcessBuilder.Redirect.INHERIT );
      // Execute the stage3 process.
      Process process3 = pb3.start();
      // Redirect the parent's System.out to be the standard input of the stage3 process.
      System.setOut( new PrintStream( process3.getOutputStream() ) );


      // Create a command line for running the stage2 child.
      ProcessBuilder pb2 = new ProcessBuilder("java", "ToUpperCase");
      // Have the stage2 child inherit the parent's (NEW) standard output stream.
      pb2.redirectOutput( ProcessBuilder.Redirect.INHERIT );
      // Execute the stage2 process.
      Process process2 = pb2.start();
      // Redirect the parent's System.out to be the standard input of the stage2 process.
      System.setOut( new PrintStream( process2.getOutputStream() ) );


      // Create a command line for running the stage1 child.
      ProcessBuilder pb1 = new ProcessBuilder("java", "RemoveVowels");
      // Have the stage1 child inherit the parent's (NEW) standard output stream.
      pb1.redirectOutput( ProcessBuilder.Redirect.INHERIT );
      // Have the stage1 child inherit the parent's standard input stream.
      pb1.redirectInput( ProcessBuilder.Redirect.INHERIT );
      // Execute the stage1 process.
      Process process1 = pb1.start();


      // Wait for the third stage child to finish its work.
      process3.waitFor();  // this throws InterruptedException
   }
}